#include "game.h"

extern int gameState;
bool tutorialShow;

const float TEXT_BOX_FULLY_VISIBLE = 750;
float textBoxTimer = 0;

const float timer_max = 500;
float timerFeedback;

extern const int LEVEL_COUNT = 25;
extern const char *levelNames[LEVEL_COUNT];
extern int currentLevel;

bool wasRestarted;
const int MAX_SHAPE_COUNT = 50;
int shapeCount;
Shape **shapes;

static int tryCount;


const char *feedbackSuccess[] =
{
	"Awesome!",
	"Great!",
	"Well done!",
};

// not actually used..
const char *feedbackFail[] =
{
	"Better luck next time",
	"Try harder!",
	"Stuck? You can select a level to play from level selection!"
};

void drawFeedback(bool complete)
{
	float centerx = OPENGL_WIDTH>>1;
	float centery = OPENGL_HEIGHT>>1;

	glColor3f(0.0,0.0,0.0);
	glDisable(GL_BLEND);
	timerFeedback+=deltatime;
	float c = min(timer_max,timerFeedback)/timer_max;
	float maxw = OPENGL_WIDTH;
	float maxh = 60*c;
	glBegin(GL_QUADS);
		glVertex2f(centerx-maxw, centery-maxh);
		glVertex2f(centerx+maxw, centery-maxh);
		glVertex2f(centerx+maxw, centery+maxh);
		glVertex2f(centerx-maxw, centery+maxh);
	glEnd();
	glColor3f(1,1,1);
	glEnable(GL_BLEND);

	glBegin(GL_LINES);
		glVertex2f(centerx-maxw, centery-maxh);
		glVertex2f(centerx+maxw, centery-maxh);
		glVertex2f(centerx-maxw, centery+maxh);		
		glVertex2f(centerx+maxw, centery+maxh);
	glEnd();

	if(c>0.99f)
	{
		if(complete)
			printCenteredX(feedbackSuccess[tryCount%3], centerx, centery);
		else
			printCenteredX(feedbackFail[tryCount%3], centerx, centery);
	}
}

bool runInEditor;

const float GOAL_TIME = 500;
int explodedCount;
int levelGoalScore;
int activePercentage;
bool allPiecesStopped;
bool playerClicked;

struct MapData 
{
	int goalScore;
	int type[MAX_SHAPE_COUNT];
	float posx[MAX_SHAPE_COUNT];
	float posy[MAX_SHAPE_COUNT];
	float size[MAX_SHAPE_COUNT];
};

MapData data;

void startGamePlay(int level)
{
	wasRestarted=false;
	timerFeedback =0;
	tryCount++;
	playerClicked = false;
	killParticles();
	tutorialShow = true;
	textBoxTimer = 0;
	explodedCount = 0;
	allPiecesStopped = false;
	// load level!
	runInEditor = false;
	
	// Otherwise its the editor..
	if(level>=0)
	{
		char buf[50];
		sprintf(buf, "data/level%d.mp", (currentLevel+1));
		loadMap(buf);
	}


}


void initLevelData()
{
	activePercentage = 50;
	int i;
	shapeCount =0;
	shapes = new Shape*[MAX_SHAPE_COUNT];
	for(i=0; i<MAX_SHAPE_COUNT; i++)
	{
		shapes[i] = new Shape;
	}
}

void restartLevel()
{
	int i;
	killParticles();
	
	for(i=0; i<15; i++)
	{
		addShape(Vector(rand()%OPENGL_WIDTH, rand()%OPENGL_HEIGHT), TYPE_TRIANGLE+rand()%TYPE_COUNT, 30);
	}
	tutorialShow = true;
}

void addShape(Vector pos, int type, int size)
{
	if(shapeCount >= MAX_SHAPE_COUNT) 
	{
		MessageBox(0,"Level cannot containg more than 50 shapes", "Error", 0);
		return;
	}
	shapes[shapeCount]->init(pos, type, size);
	
	shapeCount++;
}

void freeGameResources()
{
	for(int i=0; i<shapeCount; i++)
	{
		delete shapes[i];
	}
	delete [] shapes;
}

void drawGame()
{
	int i;

	drawBackground();
	
	glColor3f(1,1,1);

	for(i=0; i<shapeCount; i++)
	{
		shapes[i]->draw();
	}
	
	for(i=0; i<shapeCount; i++)
	{
		shapes[i]->drawAimHelper();
	}
		
	drawParticles();

	if(gameState == 4 || (gameState==5 && runInEditor))
	{
		if(tutorialShow) 
			drawTextBox();
		else
		{
			glColor3f(1,1,1);
			if(explodedCount >= levelGoalScore) glColor3f(0.6f, 0.6f, 0.98f);
			// hud
			char buf[50];
			sprintf(buf, "Exploded: %d / %d out of total %d", explodedCount, levelGoalScore, shapeCount);
			print(buf, 10, 15);
		}

		if(explodedCount == shapeCount)
		{
			drawFeedback((explodedCount >= levelGoalScore));
		}

	}
}	

bool updateGame(float delta, bool processinput)
{
	if(tutorialShow)	
	{
		updateTextBox(delta);
		if(mouseLeft || keyEnter)
		{
			mouseLeft = false;
			keyEnter = false;
			tutorialShow=false;
		}
		return false;
	}
	else
	{
		if(playerClicked)
		{
			if(allPiecesStopped)
			{
				if(explodedCount >= levelGoalScore)
				{
					// advance level!
					currentLevel++;
					if(currentLevel>LEVEL_COUNT-1)
					{
						return true;
					}
					startGamePlay(currentLevel);
					
				}
				else
				{
					startGamePlay(currentLevel);
					wasRestarted=true;
				}
			}
		}
	}

		int i,j;
		explodedCount = 0;
	//	if(mouseLeft)
		{
			allPiecesStopped = true;
			//addParticle(0, mouse);
			for(i=0; i<shapeCount; i++)
			{
				
				if(!playerClicked || (shapes[i]->exploding && shapes[i]->explosiondistance < MAX_D))
				{
					allPiecesStopped = false;
				}
				
					

				if(shapes[i]->exploding) 
				{
					explodedCount++;
					continue;
				}
	
				shapes[i]->showLines = false;
	
				if(!playerClicked)
				{
					if(shapes[i]->mouseTest(mouse.x, mouse.y))
					{
						Shape *p = shapes[i];
						
						shapes[i]->showLines = true;
		
						if(mouseLeft)
						{
							for(int k=0; k<p->type+3; k++)
							{
								//addParticlesOnLine(p->lines[k].start, p->lines[k].end, 100);
								addParticlesOnLine3(p->lines[k].start, p->lines[k].start+p->lines[k].dir*1500, 60);
								addParticlesOnLine3(p->lines[k].end, p->lines[k].end+p->lines[k].dir*1500, 60);
							}

							sound->play(4);
		
							shapes[i]->exploding = true;
							playerClicked = true;
						}
					}
				}
			}
		}
	
		//if(mouseRight) restartLevel();
	
		for(i=0; i<shapeCount; i++)
		{
			for(j=0; j<shapeCount; j++)
			{
				if(i==j) continue;
	
				if(shapes[i]->collisionTestRequired(shapes[j]))
				{
					if(shapes[i]->testCollision(shapes[j]))
					{
						int n = shapes[i]->exploding ? j : i;
						int k;
						for(k=0; k<shapes[n]->type+3; k++)
						{
							addParticlesOnLine2(shapes[n]->lines[k].start, shapes[n]->lines[k].end, 30);
						}
	
						sound->play(min(shapes[i]->type, 6));
						shapes[i]->exploding = true;
						shapes[j]->exploding = true;
					}
				}
			}
		}
	
		for(i=0; i<shapeCount; i++)
		{
			shapes[i]->update(deltatime, i);
		}

		if(keyEscape) 
		{
			keyEscape=false;
			return true;
		}
	
	
		//updateParticles(deltatime);


		return false;
}	

// -----------------------
// Shapes

void Shape::updateLines()
{
	delete [] lines;
	lines = new Line[3+type];
}

void Shape::draw()
{
	// Do not draw if not on screen
	if(explosiondistance > MAX_D) return;

	int n;
		for(n=0; n<3+type; n++)
		{
			
			if(exploding)
			{
				float dd = min(explosiontimer, 10)/10.0f;
				dd = dd*dd;
				float len = 40.0f * dd*dd;
				glBegin(GL_QUADS);
					glColor4f(1,1,1,0.8f);
					glVertex2f(lines[n].end.x,							lines[n].end.y);
					glVertex2f(lines[n].start.x,						lines[n].start.y);
					glColor4f(1,1,1,0.01f);
					glVertex2f(lines[n].start.x -lines[n].dir.x*len,	lines[n].start.y-lines[n].dir.y*len);
					glVertex2f(lines[n].end.x	-lines[n].dir.x*len,	lines[n].end.y	-lines[n].dir.y*len);
				glEnd();
				glColor3f(1,1,1);
			}
			
			
			glBegin(GL_LINES);			
			glVertex2f(lines[n].start.x, lines[n].start.y);
			glVertex2f(lines[n].end.x, lines[n].end.y);
			glEnd();
		}
}

void Shape::update(int delta, int idn)
{
	// Do not update if not on screen
	if(explosiondistance > MAX_D) return;

	if(exploding)
	{
		explosiontimer += delta*0.01f;
		explosiondistance += delta*0.15f * min(explosiontimer, 10)/10.0f;
	}
	else
	{
		angle += delta*anglespeed;
	}
	

	//size += delta*0.01f;
	const float FULL_ANGLE = 6.2831852f;
	float angleinc = FULL_ANGLE / (3+type);
	int n;
	float tmpangle = angle;
	for(n=0; n<3+type; n++)
	{
		lines[n].start.x = pos.x + sinf(tmpangle)*size;
		lines[n].start.y = pos.y + cosf(tmpangle)*size;
		lines[n].end.x = pos.x + sinf(tmpangle+angleinc)*size;
		lines[n].end.y = pos.y + cosf(tmpangle+angleinc)*size;
		Vector d = Vector(sinf(tmpangle+angleinc*0.5f), cosf(tmpangle+angleinc*0.5f));
		lines[n].dir = d;

		if(exploding)
		{
			d*=explosiondistance;
			lines[n].start += d;
			lines[n].end += d;
		}
		tmpangle+=angleinc;
	}

	
	if(!exploding)
	{
		if(currentLevel > 22)
		{
			if(idn == 5)
			{
				float t = gametime*0.001f;
				pos = origpos + Vector(cosf(t)*120.0f, 0);
			}

		}

		if(currentLevel > 13)
		{
			if(idn == 0)
			{
				float t = gametime*0.001f;
				pos = origpos + Vector(cosf(t)*30.0f, sinf(t)*40.0f);
			}
		}

		if(currentLevel > 19)
		{
			if(idn == 10)
			{
				float t = -gametime*0.0015f;
				pos = origpos + Vector(cosf(t)*60.0f, sinf(t)*60.0f);
			}
		}
	}
};

// -----------------------
// Lines and collisions

bool Line::lineIntersection(Line *l)
{
	float denom = ((l->end.y - l->start.y)*(end.x - start.x)) - ((l->end.x - l->start.x)*(end.y - start.y));
	
	if(denom == 0.0f)
	{
		return false;
	}

	float nume_a = ((l->end.x - l->start.x)*(start.y - l->start.y)) - ((l->end.y - l->start.y)*(start.x - l->start.x));
	float nume_b = ((end.x - start.x)*(start.y - l->start.y)) - ((end.y - start.y)*(start.x - l->start.x));
	float ua = nume_a / denom;
	float ub = nume_b / denom;
	return(ua >= 0.0f && ua <= 1.0f && ub >= 0.0f && ub <= 1.0f);
}

// -----------------------
// Background

void drawBackground()
{
	const float w = OPENGL_WIDTH;
	const float h = OPENGL_HEIGHT;

	glColor3f(0.15f, 0.15f, 0.15f);

	int t = gametime;

	const float blocksize = 24.0f;
	float sini = abs(sinf(3.1415926f*t*0.0001f)*blocksize);
	float cosi = abs(cosf(3.1415926f*t*0.0001f)*blocksize);

	float x,y;
	for(x=0; x<w; x+=blocksize)
	for(y=0; y<w; y+=blocksize)
	{
		glBegin(GL_LINE_STRIP);
			glVertex2f(x,y+cosi);
			glVertex2f(x+blocksize,y+sini);
		glEnd();
	}
}


void drawBackground2()
{
	const float w = OPENGL_WIDTH;
	const float h = OPENGL_HEIGHT;

	glColor3f(0.106135f, 0.164175f, 0.23185215f);
	

	int t = gametime;

	const float blocksize = 44.0f;
	float sini = abs(sinf(3.1415926f*t*0.0001f)*blocksize);
	float cosi = abs(cosf(3.1415926f*t*0.0001f)*blocksize);
	float sini2 = powf(abs(cosf(3.1415926f*t*0.0011f)),6)*blocksize;
	float cosi2 = powf(abs(cosf(3.1415926f*t*0.0011f)),12)*blocksize;

	float x,y;
	for(x=0; x<w; x+=blocksize)
	for(y=0; y<w; y+=blocksize)
	{
		glBegin(GL_LINES);
			glVertex2f(x,y+cosi);
			glVertex2f(x,y+sini);
			
			glVertex2f(x+sini,y+cosi2);
			glVertex2f(x+cosi,y+sini2);
		glEnd();
	}
	/*
	for(x=0; x<w; x+=blocksize)
	{
		glBegin(GL_QUAD_STRIP);
		for(y=0; y<w; y+=blocksize*0.2f)
		{
			float cosi = sinf(t*0.0000230f*(((int)(y*y+5))%35))*24.0f;
			glVertex2f(x+cosi,y);
			glVertex2f(x+cosi+blocksize*0.5f,y);
		}
		glEnd();
	}*/
	glLineWidth(3);

	drawParticles();
	glColor3f(1,1,1);
}


// -----------------------
// Particle system

int particleCount;
Particle **particles;


void Particle::update(int deltatime)
{
	pos += spd * deltatime;
	time+=deltatime;
	if(time > killtime) alive = false;
}


void Particle::init(int t, Vector p)
{
	pos = p;
	type = t;
	size = 10;
	alive = true;
	time = 0;

	switch(type)
	{
	case PARTICLE_FLARE_CHAIN_3:
	{
		size*=0.5f;
		float len = 0.016f;
		spd = Vector( (randFloat()-0.5f) * len, (randFloat()-0.5f) * len );
		killtime = 700 + randFloat()*100.0f;			
	}
	break;

	case PARTICLE_FLARE_CHAIN:
	case PARTICLE_FLARE:
	{
	
		float len = 0.15f;
		spd = Vector( (randFloat()-0.5f) * len, (randFloat()-0.5f) * len );
		killtime = 500 + randFloat()*500.0f;			

	} break;
	default: 
		break;
	}
}

void Particle::draw()
{
	RRect to;
	to.x = pos.x - (size>>1); 
	to.y = pos.y - (size>>1);
	to.w = size;
	to.h = size;

	
	float fadetime = 400.0f;
	float c = 1;
	if(time > killtime - fadetime)
	{
		c = 1-(time-(killtime-fadetime))/fadetime;
	}
	glColor3f(c,c,c);

	switch(type)
	{
	case PARTICLE_FLARE:

		drawSprite(to, texture, 0.01f, 1.0f);

		break;

	case PARTICLE_FLARE_CHAIN_3:
	case PARTICLE_FLARE_CHAIN:

		drawSprite(to, texture2, 0.01f, 1.0f);

		break;
	default: break;
	}
}

void initParticles()
{
	particleCount = 2000;
	particles = new Particle*[particleCount];
	for(int i=0; i<particleCount; i++)
	{
		particles[i] = new Particle();
	}
}

void freeParticles()
{
	particleCount = 2000;
	for(int i=0; i<particleCount; i++)
	{
		delete particles[i];
	}
	delete [] particles;
}


void updateParticles(int deltatime)
{
	int n;
	for(n=0; n<particleCount; n++)
	{
		if(particles[n]->alive)
		{
			particles[n]->update(deltatime);
		}
	}
}

void drawParticles()
{

	glEnable(GL_TEXTURE_2D);
	
	int n;
	for(n=0; n<particleCount; n++)
	{
		if(particles[n]->alive)
		{
			particles[n]->draw();
		}
	}

	glDisable(GL_TEXTURE_2D);
	glBlendFunc(GL_SRC_ALPHA, GL_ONE);
}

void addParticle(int type, Vector pos)
{
	int n;
	for(n=0; n<particleCount; n++)
	{
		if(!particles[n]->alive)
		{
			particles[n]->init(type, pos);
			return;
		}
	}
}

void killParticles()
{
	int n;
	for(n=0; n<particleCount; n++)
	{
		particles[n]->alive = false;
	}
}

void addParticlesOnLine(Vector a, Vector b, int n)
{
	Vector dir = (b-a);// * (1/(float)n);
	float len = dir.length();
	dir.normalize();
	Vector d = dir*len/(float(n));

	int i;
	for(i=0; i<n; i++)
	{
		addParticle(PARTICLE_FLARE, a);
		a += d;
	}
}
void addParticlesOnLine2(Vector a, Vector b, int n)
{
	Vector dir = (b-a);// * (1/(float)n);
	float len = dir.length();
	dir.normalize();
	Vector d = dir*len/(float(n));

	int i;
	for(i=0; i<n; i++)
	{
		addParticle(PARTICLE_FLARE_CHAIN, a);
		a += d;
	}
}

void addParticlesOnLine3(Vector a, Vector b, int n)
{
	Vector dir = (b-a);// * (1/(float)n);
	float len = dir.length();
	dir.normalize();
	Vector d = dir*len/(float(n));

	int i;
	for(i=0; i<n; i++)
	{
		addParticle(PARTICLE_FLARE_CHAIN_3, a);
		a += d;
	}
}

// -------
// util 
float randFloat() { return rand()%1000*0.001f; }

void drawTextBox()
{
	float f = textBoxTimer/TEXT_BOX_FULLY_VISIBLE;

	float smod = 0.135f;
	if(gameState!=4) smod = 0.24f;

	float maxw = OPENGL_WIDTH  * smod * 1.5f  * f;
	float maxh = OPENGL_HEIGHT * smod * f;
	if(wasRestarted) maxw = OPENGL_WIDTH * 0.4f * f;	
	float centerx = OPENGL_WIDTH>>1;
	float centery = OPENGL_HEIGHT>>1;

	glColor3f(0.0,0.0,0.0);
	glDisable(GL_BLEND);
	glBegin(GL_QUADS);
		glVertex2f(centerx-maxw, centery-maxh);
		glVertex2f(centerx+maxw, centery-maxh);
		glVertex2f(centerx+maxw, centery+maxh);
		glVertex2f(centerx-maxw, centery+maxh);
	glEnd();
	glEnable(GL_BLEND);

	glColor3f(1,1,1);
	glLineWidth(2);
	glBegin(GL_LINE_STRIP);
		glVertex2f(centerx-maxw, centery-maxh);
		glVertex2f(centerx+maxw, centery-maxh);
		glVertex2f(centerx+maxw, centery+maxh);
		glVertex2f(centerx-maxw, centery+maxh);
		glVertex2f(centerx-maxw, centery-maxh);
	glEnd();

	const int mod = 5;
	glBegin(GL_LINE_STRIP);
		glVertex2f(centerx-maxw+mod, centery-maxh+mod);
		glVertex2f(centerx+maxw-mod, centery-maxh+mod);
		glVertex2f(centerx+maxw-mod, centery+maxh-mod);
		glVertex2f(centerx-maxw+mod, centery+maxh-mod);
		glVertex2f(centerx-maxw+mod, centery-maxh+mod);
	glEnd();
	glLineWidth(3);

	// Draw text here
	if(f>0.99f)
	{
		if(gameState==4)
		{
			char buf[200];
			if(wasRestarted)
			{
				printCenteredX(feedbackFail[tryCount%3], centerx, centery-25);
			}
			else 
			{
				sprintf(buf, "%d: %s", (1+currentLevel), levelNames[currentLevel]);
				printCenteredX(buf, centerx, centery-25);
			}
			
			sprintf(buf, "%s%d", "Level goal score: ", levelGoalScore);
			printCenteredX(buf, centerx, centery+25);
		}
		else
		{
			int y = centery-80;
			printCenteredX("Level editor instructions", centerx, y+=15);
			printCenteredX("", centerx, y+=15);
			printCenteredX("Mouse click adds new shape", centerx, y+=15);
			printCenteredX("Arrows cycle through shape properties", centerx, y+=15);
			printCenteredX("O opens new level", centerx, y+=15);
			printCenteredX("S saves old level", centerx, y+=15);
			printCenteredX("Q/W change goal shape score", centerx, y+=15);
			printCenteredX("R allows you to play the level in the game", centerx, y+=15);
			printCenteredX("Escape gets back to menu", centerx, y+=15);
			printCenteredX("N starts new map", centerx, y+=15);
			printCenteredX("", centerx, y+=15);
			printCenteredX("Please use the editor in windowed mode.", centerx, y+=15);

		}
	}	
	
}

void updateTextBox(float delta)
{
	textBoxTimer += delta;
	
	if(textBoxTimer<0) textBoxTimer = 0;
	if(textBoxTimer>TEXT_BOX_FULLY_VISIBLE) textBoxTimer = TEXT_BOX_FULLY_VISIBLE;
}

// editor
// ----------------------------

void packMap()
{
	data.goalScore = levelGoalScore;

	int n;
	for(n=0; n<MAX_SHAPE_COUNT; n++)
	{
		if(n < shapeCount)
		{
			Shape *p = shapes[n];
			data.type[n] = p->type;
			data.posx[n] = p->pos.x;
			data.posy[n] = p->pos.y;
			data.size[n] = p->size;
		}
		else 
		{
			data.type[n] = -1;
			data.posx[n] = -1;
			data.posy[n] = -1;
			data.size[n] = -1;
		}

	}
}

void unpackMap()
{
	shapeCount = 0;
	levelGoalScore = data.goalScore;

	int n;
	for(n=0; n<MAX_SHAPE_COUNT; n++)
	{
		if(data.type[n] != -1)
		{
			addShape(Vector(data.posx[n], data.posy[n]), data.type[n], data.size[n]);
		}
	}
}


void saveMap(const char *filename)
{
	packMap();

	std::fstream binary_file(filename,std::ios::out|std::ios::binary);
    binary_file.write(reinterpret_cast<char *>(&data),sizeof(MapData));
    binary_file.close();
}

void loadMap(const char *filename)
{
	std::fstream binary_file(filename,std::ios::binary|std::ios::in);
    binary_file.read(reinterpret_cast<char *>(&data),sizeof(MapData));
    binary_file.close();
	unpackMap();
}

const char *typeNames[] = 
{
	"TYPE_TRIANGLE",
	"TYPE_RECTANGLE",
	"TYPE_PENTAGON",
	"TYPE_HEXAGON",
	"TYPE_HEPTAGON",
	"TYPE_OCTAGON",
	"TYPE_NONAGON",
	"TYPE_DECADON",
};

int selectedBlock = -1;
int selectedProperty;

const int PROPERTY_COUNT = 3;
const char *properties[] = { "Type", "Size", "Rotatespeed" };

void drawEditor()
{
	if(runInEditor)
	{
		drawGame();
		return ;
	}



	glColor3f(1,1,1);
	int y = OPENGL_HEIGHT-10;
	printSmall("Active property: ", 10 , y);
	printSmall(properties[selectedProperty], 120 , y);

	char buf[250];

	if(selectedBlock >= 0)
	{
		if(selectedProperty==0)
		{
			sprintf(buf, "%s", typeNames[shapes[selectedBlock]->type]);
		} else if(selectedProperty==1)
		{
			sprintf(buf, "%d", shapes[selectedBlock]->size);
		} else if(selectedProperty==2)
		{
			sprintf(buf, "%f", shapes[selectedBlock]->anglespeed);
		}

		printSmall(buf, 200 , y);
	}

	print("LEVEL EDITOR", 910, 15);

	sprintf(buf, "Shapes: %d", shapeCount);
	printSmall(buf, 10, 15);

	sprintf(buf, "Goal count: %d", levelGoalScore);
	printSmall(buf, 100, 15);

	sprintf(buf, "Current percentage of total: %d  | Q & W change", activePercentage);
	printSmall(buf, 210, 15);

	drawGame();

	if(selectedBlock>=0)
	{
		glColor3f(0.76f, 0.6f, 0.50f);
		glBegin(GL_LINE_STRIP);
			glVertex2f(shapes[selectedBlock]->pos.x - shapes[selectedBlock]->size, shapes[selectedBlock]->pos.y - shapes[selectedBlock]->size);
			glVertex2f(shapes[selectedBlock]->pos.x - shapes[selectedBlock]->size, shapes[selectedBlock]->pos.y + shapes[selectedBlock]->size);
			glVertex2f(shapes[selectedBlock]->pos.x + shapes[selectedBlock]->size, shapes[selectedBlock]->pos.y + shapes[selectedBlock]->size);
			glVertex2f(shapes[selectedBlock]->pos.x + shapes[selectedBlock]->size, shapes[selectedBlock]->pos.y - shapes[selectedBlock]->size);
			glVertex2f(shapes[selectedBlock]->pos.x - shapes[selectedBlock]->size, shapes[selectedBlock]->pos.y - shapes[selectedBlock]->size);
		glEnd();
		glColor3f(1,1,1);
	}

	if(tutorialShow) drawTextBox();
}

void updateProperty(int n)
{
	if(selectedBlock >= 0)
	{
		if(selectedProperty==0)
		{
			keyRight = false;
			keyLeft = false;
			mouseRight = false;
			shapes[selectedBlock]->type+=n;
			if(shapes[selectedBlock]->type<0) shapes[selectedBlock]->type = TYPE_COUNT-1;
			if(shapes[selectedBlock]->type>TYPE_COUNT-1) shapes[selectedBlock]->type = 0;
			shapes[selectedBlock]->updateLines();
			updateGame(0);
		} else if(selectedProperty==1)
		{
			keyRight = false;
			keyLeft = false;
			mouseRight = false;
			//sprintf(buf, "%f", );
			shapes[selectedBlock]->size += n*4.0f;
			if(shapes[selectedBlock]->size < 1.0f) shapes[selectedBlock]->size = 1.0f;
		} 
		else if(selectedProperty==2)
		{
			shapes[selectedBlock]->anglespeed += n*0.00002f;
		}
	}
}

bool updateEditor(float delta)
{

	if(runInEditor)
	{
		if(updateGame(delta))
		{
			runInEditor=false;
			loadMap("test.mp");
		}

		return false;
	}


	levelGoalScore = max(shapeCount * activePercentage / 100, 2);

	int i;
	
	if(tutorialShow) 
	{
		textBoxTimer = TEXT_BOX_FULLY_VISIBLE;
		if(mouseLeft || keyEnter) 
		{
			tutorialShow = false;
			mouseLeft = false;
			keyEnter = false;
		}

		return false;
	}
	

	for(i=0; i<shapeCount; i++)
	{
		shapes[i]->update(deltatime, i);
	}

	if(keyUp) 
	{
		selectedProperty--;	
		keyUp = false;
	}
	if(keyDown) 
	{
		selectedProperty++;
		keyDown = false;
	}
	if(selectedProperty<0) selectedProperty = PROPERTY_COUNT-1;
	if(selectedProperty>PROPERTY_COUNT-1) selectedProperty = 0;

	if(keySpace && selectedBlock >= 0 && selectedProperty == 2)
	{
		shapes[selectedBlock]->anglespeed = 0;
	}

	if(keyLeft)
	{
		updateProperty(-1);
	}
	if(keyRight || mouseRight)
	{
		updateProperty(1);
	}

	if(keyN)
	{
		keyN=false;
		if(MessageBox(0,"New map?", "Boostcraft", MB_YESNO )==IDYES) 
		{
			shapeCount = 0;
			selectedBlock = -1;
		}
	}

	if(keyQ)
	{
		keyQ=false;
		activePercentage -= 5;
		if(activePercentage < 0) activePercentage = 0;
		levelGoalScore = shapeCount * activePercentage / 100;
	}

	if(keyW)
	{
		keyW=false;
		activePercentage += 5;
		if(activePercentage > 100) activePercentage = 100;
		levelGoalScore = shapeCount * activePercentage / 100;
		
	}

	if(keyR)
	{	
		keyR = false;
		if(shapeCount < 2)
		{
			MessageBox(0, "Shape count must be atleast 2!\nAdd some shapes with left mouse button.", "Error", 0);
		}
		else
		{
			startGamePlay(-1);
			saveMap("temp.mp");
			loadMap("temp.mp");
			runInEditor = true;
			tutorialShow = false;
		}
	}

	if(keyO)
	{
		keyO=false;
		std::string fname = openDialog(".mp");
		if(fname!="")
		{
			loadMap(fname.c_str());
		}
	}

	if(keyS)
	{
		
		keyS=false;

		if(shapeCount < 2)
		{
			MessageBox(0, "Shape count must be atleast 2!\nAdd some shapes with left mouse button.", "Error", 0);
		}
		else
		{
			std::string fname = saveDialog(".mp");
			if(fname!="")
			{
				saveMap(fname.c_str());
			}
		}
	}

	if(mouseLeft)
	{
		int n= findSelecterShape();
		if(n>=0) 
		{
			if(selectedBlock == n)
			{
				selectedProperty++;
				if(selectedProperty>PROPERTY_COUNT-1) selectedProperty = 0;
			}
			else
			{
				selectedBlock = n;
			}
			mouseLeft = false;
		}
		else 
		{
			mouseLeft = false;
			addShape(mouse, TYPE_TRIANGLE, 30.0f);
			selectedBlock = shapeCount-1;
		}
	}


	if(keyEscape) return true;
	return false;
}

int findSelecterShape()
{
	int i;
	for(i=0; i<shapeCount; i++) shapes[i]->showLines = false;
	for(i=0; i<shapeCount; i++)
	{
		if(shapes[i]->mouseTest(mouse.x, mouse.y))
		{
			shapes[i]->showLines = true;
			return i;
		}
	}
	return -1;
}


std::string openDialog(const char *file_exts)
{
    OPENFILENAME ofn;
    char szFile[260];
    //ZeroMemory(&ofn, sizeof(ofn));
	memset(&ofn, 0, sizeof(ofn));
    ofn.lStructSize = sizeof(ofn);
    ofn.hwndOwner = NULL;//(HWND)getHWND();//hWnd;
    ofn.lpstrFile = szFile;
    ofn.lpstrFile[0] = '\0';
    ofn.nMaxFile = sizeof(szFile);
    ofn.lpstrFilter = file_exts;
    ofn.nFilterIndex = 1;
    ofn.lpstrFileTitle = NULL;
    ofn.nMaxFileTitle = 0;
    ofn.lpstrInitialDir = NULL;
    ofn.Flags = OFN_PATHMUSTEXIST | OFN_FILEMUSTEXIST | OFN_HIDEREADONLY | OFN_OVERWRITEPROMPT;
	
	if(GetOpenFileName(&ofn)!=0)
		return ofn.lpstrFile;
	else 
		return "";
}


std::string saveDialog(const char *file_exts)
{
    OPENFILENAME ofn;
    char szFile[260];
    ZeroMemory(&ofn, sizeof(ofn));
    ofn.lStructSize = sizeof(ofn);
    ofn.hwndOwner = NULL;//(HWND)getHWND();//hWnd;
    ofn.lpstrFile = szFile;
    ofn.lpstrFile[0] = '\0';
    ofn.nMaxFile = sizeof(szFile);
    ofn.lpstrFilter = file_exts;
    ofn.nFilterIndex = 1;
    ofn.lpstrFileTitle = NULL;
    ofn.nMaxFileTitle = 0;
    ofn.lpstrInitialDir = NULL;
    ofn.Flags = OFN_PATHMUSTEXIST | OFN_FILEMUSTEXIST | OFN_HIDEREADONLY | OFN_OVERWRITEPROMPT;
	
	if(GetSaveFileName(&ofn)!=0)
		return ofn.lpstrFile;
	else 
		return "";
}
